// Copyright (c) 2013 The Chromium Embedded Framework Authors. All rights
// reserved. Use of this source code is governed by a BSD-style license that
// can be found in the LICENSE file.

package Browser;

import org.cef.CefApp;
import org.cef.CefApp.CefVersion;
import org.cef.CefClient;
import org.cef.CefSettings;
import org.cef.browser.CefBrowser;
import org.cef.browser.CefFrame;
import org.cef.browser.CefMessageRouter;
import org.cef.callback.CefCommandLine;
import org.cef.handler.CefDisplayHandlerAdapter;
import org.cef.handler.CefFocusHandlerAdapter;
import org.cef.handler.CefLoadHandlerAdapter;
import org.cef.network.CefCookieManager;

import App.MngSetting;

import java.awt.BorderLayout;
import java.awt.KeyboardFocusManager;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;

import javax.imageio.ImageIO;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import Browser.dialog.DownloadDialog;
import Browser.handler.AppHandler;
import Browser.handler.ContextMenuHandler;
import Browser.handler.DragHandler;
import Browser.handler.JSDialogHandler;
import Browser.handler.KeyboardHandler;
import Browser.handler.MessageRouterHandler;
import Browser.handler.MessageRouterHandlerEx;
import Browser.handler.RequestHandler;
import Browser.ui.ControlPanel;
import Browser.ui.MenuBar;
import Browser.ui.StatusPanel;
import Browser.util.DataUri;

public class MainFrame extends BrowserFrame {
    private static final long serialVersionUID = 1L;
    public CefCookieManager cookieManager = null;
    public static CefVersion version;
    public static String userAgent;
	
    public static void main(String[] args) throws IOException, URISyntaxException {
        // Perform startup initialization on platforms that require it.
        if (!CefApp.startup(args)) {
            System.out.println("Startup CefApp initialization failed!");
            return;
        }else{
        	System.out.println("Startup CefApp initialization success!");
        }

        // OSR mode is enabled by default on Linux.
        // and disabled by default on Windows and Mac OS X.
        boolean osrEnabledArg = false;
        boolean transparentPaintingEnabledArg = false;
        boolean createImmediately = false;
        for (String arg : args) {
            arg = arg.toLowerCase();
            if (arg.equals("--off-screen-rendering-enabled")) {
                osrEnabledArg = true;
            } else if (arg.equals("--transparent-painting-enabled")) {
                transparentPaintingEnabledArg = true;
            } else if (arg.equals("--create-immediately")) {
                createImmediately = true;
            }
        }

        System.out.println("Offscreen rendering " + (osrEnabledArg ? "enabled" : "disabled"));

        // MainFrame keeps all the knowledge to display the embedded browser
        // frame.
        final MainFrame frame = new MainFrame(
                osrEnabledArg, transparentPaintingEnabledArg, createImmediately, args);
        frame.setSize(718, 700);
        frame.setResizable(false);
        frame.setVisible(true);
        frame.setLocationRelativeTo(null);
    }

    private final CefClient client_;
    private String errorMsg_ = "";
    private ControlPanel control_pane_;
    private StatusPanel status_panel_;
    private boolean browserFocus_ = true;
    private boolean osr_enabled_;
    private boolean transparent_painting_enabled_;

    public MainFrame(boolean osrEnabled, boolean transparentPaintingEnabled,
            boolean createImmediately, String[] args) throws IOException, URISyntaxException {
        this.osr_enabled_ = osrEnabled;
        this.transparent_painting_enabled_ = transparentPaintingEnabled;
        // Custom User Agent
        String verOS = " (" + System.getProperty("os.name") + "; " + System.getProperty("os.arch") + ")";
        String verJava = " Java/" + System.getProperty("java.runtime.version");
        MainFrame.userAgent = "WP-WebRTC2-client" + verOS + verJava;

        CefApp myApp;
        if (CefApp.getState() != CefApp.CefAppState.INITIALIZED) {
            // 1) CefApp is the entry point for JCEF. You can pass
            //    application arguments to it, if you want to handle any
            //    chromium or CEF related switches/attributes in
            //    the native world.
            CefSettings settings = new CefSettings();
            settings.user_agent = MainFrame.userAgent;
            settings.windowless_rendering_enabled = osrEnabled;
            settings.cache_path = this.getCookiesPath();
            settings.background_color = settings.new ColorType(100, 255, 242, 211);
            myApp = CefApp.getInstance(args, settings);

            version = myApp.getVersion();
            System.out.println("Using:\n" + version);

            //    We're registering our own AppHandler because we want to
            //    add an own schemes (search:// and client://) and its corresponding
            //    protocol handlers. So if you enter "search:something on the web", your
            //    search request "something on the web" is forwarded to www.google.com
            CefApp.addAppHandler(new AppHandler(args){
                @Override
                public void onBeforeCommandLineProcessing(String process_type, CefCommandLine command_line) {
                    super.onBeforeCommandLineProcessing(process_type, command_line);
                    if (process_type.isEmpty()) {
                        command_line.appendSwitchWithValue("enable-media-stream","true");
                        command_line.appendSwitchWithValue("use-fake-ui-for-media-stream","true");
                        command_line.appendSwitchWithValue("persist-session-cookies","true");
                    }
                }
            });    
        } else {
            myApp = CefApp.getInstance();
        }
        
        //    By calling the method createClient() the native part
        //    of JCEF/CEF will be initialized and an  instance of
        //    CefClient will be created. You can create one to many
        //    instances of CefClient.
        client_ = myApp.createClient();
        
        BufferedImage imgIcon = null;
    		
    		if (null != MainFrame.class.getResource("/src/img/videochat.png")) {
    			imgIcon = ImageIO.read( 
    				MainFrame.class.getResource("/src/img/videochat.png")
    			);
    		} else {
    			imgIcon = ImageIO.read( 
    				MainFrame.class.getResource("/img/videochat.png")
    			);
    		}
    		
        setIconImage(imgIcon);
        
        // 2) You have the ability to pass different handlers to your
        //    instance of CefClient. Each handler is responsible to
        //    deal with different informations (e.g. keyboard input).
        //
        //    For each handler (with more than one method) adapter
        //    classes exists. So you don't need to override methods
        //    you're not interested in.
        DownloadDialog downloadDialog = new DownloadDialog(this);
        client_.addContextMenuHandler(new ContextMenuHandler(this));
        client_.addDownloadHandler(downloadDialog);
        client_.addDragHandler(new DragHandler());
        client_.addJSDialogHandler(new JSDialogHandler());
        client_.addKeyboardHandler(new KeyboardHandler());
        client_.addRequestHandler(new RequestHandler(this));

        //    Beside the normal handler instances, we're registering a MessageRouter
        //    as well. That gives us the opportunity to reply to JavaScript method
        //    calls (JavaScript binding). We're using the default configuration, so
        //    that the JavaScript binding methods "cefQuery" and "cefQueryCancel"
        //    are used.
        CefMessageRouter msgRouter = CefMessageRouter.create();
        msgRouter.addHandler(new MessageRouterHandler(), true);
        msgRouter.addHandler(new MessageRouterHandlerEx(client_), false);
        client_.addMessageRouter(msgRouter);

        // 2.1) We're overriding CefDisplayHandler as nested anonymous class
        //      to update our address-field, the title of the panel as well
        //      as for updating the status-bar on the bottom of the browser
        client_.addDisplayHandler(new CefDisplayHandlerAdapter() {
            @Override
            public void onAddressChange(CefBrowser browser, CefFrame frame, String url) {
                control_pane_.setAddress(browser, url);
            }
            @Override
            public void onTitleChange(CefBrowser browser, String title) {
                setTitle(title);
            }
            @Override
            public void onStatusMessage(CefBrowser browser, String value) {
                status_panel_.setStatusText(value);
            }
        });

        // 2.2) To disable/enable navigation buttons and to display a progress bar
        //      which indicates the load state of our web site, we're overloading
        //      the CefLoadHandler as nested anonymous class. Beside this, the
        //      load handler is responsible to deal with (load) errors as well.
        //      For example if you navigate to a URL which does not exist, the
        //      browser will show up an error message.
        client_.addLoadHandler(new CefLoadHandlerAdapter() {
            @Override
            public void onLoadingStateChange(CefBrowser browser, boolean isLoading,
                    boolean canGoBack, boolean canGoForward) {
                control_pane_.update(browser, isLoading, canGoBack, canGoForward);
                status_panel_.setIsInProgress(isLoading);
                if (!isLoading && !errorMsg_.isEmpty()) {
                    browser.loadURL(DataUri.create("text/html", errorMsg_));
                    errorMsg_ = "";
                }
            }

            @Override
            public void onLoadError(CefBrowser browser, CefFrame frame, ErrorCode errorCode,
                    String errorText, String failedUrl) {
                if (errorCode != ErrorCode.ERR_NONE && errorCode != ErrorCode.ERR_ABORTED) {
                    errorMsg_ = "<html><head>";
                    errorMsg_ += "<title>Error while loading</title>";
                    errorMsg_ += "</head><body>";
                    errorMsg_ += "<h1>" + errorCode + "</h1>";
                    errorMsg_ += "<h3>Failed to load " + failedUrl + "</h3>";
                    errorMsg_ += "<p>" + (errorText == null ? "" : errorText) + "</p>";
                    errorMsg_ += "</body></html>";
                    browser.stopLoad();
                }
            }
        });
        
        String strURL = ServerPolling.UserName.isEmpty() ? this.getUrlProfile() : this.getUrlVideoChat();
        if (args == null) {
        	strURL = "http://www.google.com";
        }
        // Create the browser.
        CefBrowser browser = client_.createBrowser(
        		strURL, osrEnabled, transparentPaintingEnabled, null);
        setBrowser(browser);

        // Set up the UI for this example implementation.
        JPanel contentPanel = createContentPanel();
        getContentPane().add(contentPanel, BorderLayout.CENTER);

        // Clear focus from the browser when the address field gains focus.
        control_pane_.getAddressField().addFocusListener(new FocusAdapter() {
            @Override
            public void focusGained(FocusEvent e) {
                if (!browserFocus_) return;
                browserFocus_ = false;
                KeyboardFocusManager.getCurrentKeyboardFocusManager().clearGlobalFocusOwner();
                control_pane_.getAddressField().requestFocus();
            }
        });

        // Clear focus from the address field when the browser gains focus.
        client_.addFocusHandler(new CefFocusHandlerAdapter() {
            @Override
            public void onGotFocus(CefBrowser browser) {
                if (browserFocus_) return;
                browserFocus_ = true;
                KeyboardFocusManager.getCurrentKeyboardFocusManager().clearGlobalFocusOwner();
                browser.setFocus(true);
            }

            @Override
            public void onTakeFocus(CefBrowser browser, boolean next) {
                browserFocus_ = false;
            }
        });

        if (createImmediately) browser.createImmediately();

        // Add the browser to the UI.
        contentPanel.add(getBrowser().getUIComponent(), BorderLayout.CENTER);
        
        cookieManager = CefCookieManager.getGlobalManager();
        
        MenuBar menuBar = new MenuBar(
                this, browser, control_pane_, downloadDialog, cookieManager);
        menuBar.addBookmark("Video Chat", this.getUrlVideoChat());
        menuBar.addBookmark("Profile", this.getUrlProfile());
        menuBar.addBookmarkSeparator();
        menuBar.addBookmark("support JCEF", "https://bitbucket.org/chromiumembedded/java-cef");
        menuBar.addBookmark("support CEF", "https://bitbucket.org/chromiumembedded/cef");
        menuBar.addBookmarkSeparator();
        menuBar.addBookmark("support WebRTC2 client", "https://adminkov.bcr.by/support");
        menuBar.addBookmark("support WebRTC2", "https://wordpress.org/plugins/wp-webrtc2");
        setJMenuBar(menuBar);
    }

    private JPanel createContentPanel() {
        JPanel contentPanel = new JPanel(new BorderLayout());
        control_pane_ = new ControlPanel(getBrowser());
        status_panel_ = new StatusPanel();
        contentPanel.add(control_pane_, BorderLayout.NORTH);
        contentPanel.add(status_panel_, BorderLayout.SOUTH);
        return contentPanel;
    }

    public boolean isOsrEnabled() {
        return osr_enabled_;
    }

    public boolean isTransparentPaintingEnabled() {
        return transparent_painting_enabled_;
    }
    
    public String getUrlVideoChat() throws IOException {
    	String hostUrlVideoChat = "";
    	String strSetting = MngSetting.readSettings();
	    String[] arrSetting = strSetting.split(";");
	    
	    if (arrSetting.length == 3) {
	    	hostUrlVideoChat = arrSetting[0];
	    } else {
	    	JOptionPane.showMessageDialog(null,
	    		"See item menu - Setting",
			    "Url of VideoChat - is invalid.",
			    JOptionPane.WARNING_MESSAGE);
	    }
    	
    	return hostUrlVideoChat;
    }
    
    private String getUrlProfile() throws IOException, URISyntaxException {
    	String hostUrlProfile = "";
    	String strSetting = MngSetting.readSettings();
	    String[] arrSetting = strSetting.split(";");
	    
	    if (arrSetting.length == 3) {
	    	hostUrlProfile = arrSetting[0];
	    	if(!hostUrlProfile.isEmpty()) {
		    	URI resultURI = new URI(hostUrlProfile);
		      String strDomain = resultURI.getHost();
		    	
		    	hostUrlProfile = strDomain + "/wp-admin/profile.php";
	    	}
	    } else {
	    	JOptionPane.showMessageDialog(null,
    			"See item menu - Setting",
    			"Url of Profile - is invalid.",
			    JOptionPane.WARNING_MESSAGE);
	    }
    	
    	return hostUrlProfile;
    }
    
    public static String getUrlAjax() throws IOException, URISyntaxException {
    	String hostUrlAjax = "";
    	String strSetting = MngSetting.readSettings();
	    String[] arrSetting = strSetting.split(";");
	    
	    if (arrSetting.length == 3) {
	    	hostUrlAjax = arrSetting[0];
	    	if(!hostUrlAjax.isEmpty()) {
		    	URI resultURI = new URI(hostUrlAjax);
		      String strDomain = resultURI.getHost();
		    	
		    	hostUrlAjax = strDomain + "/wp-admin/admin-ajax.php";
	    	}
	    } else {
	    	JOptionPane.showMessageDialog(null,
    			"See item menu - Setting",
    			"Url of admin-ajax.php - is invalid.",
			    JOptionPane.WARNING_MESSAGE);
	    }
    	
    	return hostUrlAjax;
    }
    
    public String getCookiesPath() throws IOException {
    	String cookiesPath = "";
    	FileInputStream fis;
    	BufferedReader br;
    	
    	try {
			fis = new FileInputStream("/settings.txt");
			br  = new BufferedReader(new InputStreamReader(fis,"UTF-8"));
			
			cookiesPath = br.readLine();
			cookiesPath = br.readLine();
	
			br.close();
    	} catch (FileNotFoundException e) {
    		JOptionPane.showMessageDialog(null,
			    e.getMessage(),
			    "Cookies Path - is invalid",
			    JOptionPane.WARNING_MESSAGE);
    	}
    	
    	return cookiesPath;
    }
}
