// Copyright (c) 2014 The Chromium Embedded Framework Authors. All rights
// reserved. Use of this source code is governed by a BSD-style license that
// can be found in the LICENSE file.

package Browser.ui;

import org.cef.OS;
import org.cef.browser.CefBrowser;
import org.cef.callback.CefRunFileDialogCallback;
import org.cef.callback.CefStringVisitor;
import org.cef.handler.CefDialogHandler.FileDialogMode;
import org.cef.network.CefCookieManager;

import java.awt.Component;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;
import java.util.Vector;

import javax.swing.JFileChooser;

import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;

import Browser.MainFrame;
import Browser.dialog.CookieManagerDialog;
import Browser.dialog.DevToolsDialog;
import Browser.dialog.DownloadDialog;
import Browser.dialog.UrlRequestDialog;
import Browser.dialog.WebPluginManagerDialog;
import Browser.util.DataUri;


@SuppressWarnings("serial")
public class MenuBar extends JMenuBar {
    class SaveAs implements CefStringVisitor {
        private PrintWriter fileWriter_;

        public SaveAs(String fName) throws FileNotFoundException, UnsupportedEncodingException {
            fileWriter_ = new PrintWriter(fName, "UTF-8");
        }

        @Override
        public void visit(String string) {
            fileWriter_.write(string);
            fileWriter_.close();
        }
    }

    private final MainFrame owner_;
    private final CefBrowser browser_;
    private String last_selected_file_ = "";
    private final JMenu bookmarkMenu_;
    private final ControlPanel control_pane_;
    private final DownloadDialog downloadDialog_;
    private final CefCookieManager cookieManager_;

    public MenuBar(MainFrame owner, CefBrowser browser, ControlPanel control_pane,
            DownloadDialog downloadDialog, CefCookieManager cookieManager) {
        owner_ = owner;
        browser_ = browser;
        control_pane_ = control_pane;
        downloadDialog_ = downloadDialog;
        cookieManager_ = cookieManager;

        setEnabled(browser_ != null);

        JMenu fileMenu = new JMenu("File");

        JMenuItem openFileItem = new JMenuItem("Open file...");
        openFileItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent arg0) {
                JFileChooser fc = new JFileChooser(new File(last_selected_file_));
                // Show open dialog; this method does not return until the dialog is closed.
                fc.showOpenDialog(owner_);
                File selectedFile = fc.getSelectedFile();
                if (selectedFile != null) {
                    last_selected_file_ = selectedFile.getAbsolutePath();
                    browser_.loadURL("file:///" + selectedFile.getAbsolutePath());
                }
            }
        });
        fileMenu.add(openFileItem);

        JMenuItem openFileDialog = new JMenuItem("Save as...");
        openFileDialog.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                CefRunFileDialogCallback callback = new CefRunFileDialogCallback() {
                    @Override
                    public void onFileDialogDismissed(
                            int selectedAcceptFilter, Vector<String> filePaths) {
                        if (!filePaths.isEmpty()) {
                            try {
                                SaveAs saveContent = new SaveAs(filePaths.get(0));
                                browser_.getSource(saveContent);
                            } catch (FileNotFoundException | UnsupportedEncodingException e) {
                                browser_.executeJavaScript("alert(\"Can't save file\");",
                                        control_pane_.getAddress(), 0);
                            }
                        }
                    }
                };
                browser_.runFileDialog(FileDialogMode.FILE_DIALOG_SAVE, owner_.getTitle(),
                        "index.html", null, 0, callback);
            }
        });
        fileMenu.add(openFileDialog);

        JMenuItem printItem = new JMenuItem("Print...");
        printItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                browser_.print();
            }
        });
        fileMenu.add(printItem);

        fileMenu.addSeparator();

        JMenuItem viewSource = new JMenuItem("View source");
        viewSource.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                browser_.viewSource();
            }
        });
        fileMenu.add(viewSource);

        fileMenu.addSeparator();

        JMenuItem showDownloads = new JMenuItem("Show Downloads");
        showDownloads.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                downloadDialog_.setVisible(true);
                downloadDialog_.setSize(740, 695);
            }
        });
        fileMenu.add(showDownloads);

        JMenuItem showCookies = new JMenuItem("Show Cookies");
        showCookies.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                CookieManagerDialog cookieManager =
                        new CookieManagerDialog(owner_, "Cookie Manager", cookieManager_);
                cookieManager.setVisible(true);
                cookieManager.setSize(740, 695);
            }
        });
        fileMenu.add(showCookies);

        JMenuItem showPlugins = new JMenuItem("Show Plugins");
        showPlugins.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                WebPluginManagerDialog pluginManager =
                        new WebPluginManagerDialog(owner_, "Plugin Manager");
                pluginManager.setVisible(true);
                pluginManager.setSize(740, 695);
            }
        });
        fileMenu.add(showPlugins);

        fileMenu.addSeparator();

        JMenuItem exitItem = new JMenuItem("Exit");
        exitItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                owner_.dispatchEvent(new WindowEvent(owner_, WindowEvent.WINDOW_CLOSING));
            }
        });
        fileMenu.add(exitItem);

        bookmarkMenu_ = new JMenu("Bookmarks");

        JMenuItem addBookmarkItem = new JMenuItem("Add bookmark");
        addBookmarkItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                addBookmark(owner_.getTitle(), control_pane_.getAddress());
            }
        });
        bookmarkMenu_.add(addBookmarkItem);
        bookmarkMenu_.addSeparator();

        JMenu serviceMenu = new JMenu("Service");

        JMenuItem showInfo = new JMenuItem("Show Info");
        String verOS = " (" + System.getProperty("os.name") + "; " + System.getProperty("os.arch") + ")";
    	String verJava = " Java/" + System.getProperty("java.runtime.version");
    	String userAgent = "WP-WebRTC2-client" + verOS + verJava;

        showInfo.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                String info = "<html><head><title>Browser info</title></head>";
                info += "<body><h1>Browser info</h1><table border=\"0\">";
                info += "<tr><td>OS:</td><td>" + System.getProperty("os.name") + "; ver." + System.getProperty("os.version") +"; " + System.getProperty("os.arch") + "</td></tr>";
                info += "<tr><td>Java version:</td><td>" + System.getProperty("java.runtime.version") + "</td></tr>";
                info += "<tr><td>Install dir JRE:</td><td>" + System.getProperty("java.home") + "</td></tr>";
                info += "<tr><td>JRE vendor:</td><td>" + System.getProperty("java.vendor") + "</td></tr>";
                info += "<tr><td>JRE URL:</td><td>" + System.getProperty("java.vendor.url") + "</td></tr>";
                info += "<tr><td>JRE version:</td><td>" + System.getProperty("java.version") + "</td></tr>";
                info += "<tr><td>User Agent:</td><td>" + userAgent + "</td></tr>";
                info += "<tr><td>CanGoBack</td><td>" + browser_.canGoBack() + "</td></tr>";
                info += "<tr><td>CanGoForward</td><td>" + browser_.canGoForward() + "</td></tr>";
                info += "<tr><td>Url last visit</td><td>" + browser_.getURL() + "</td></tr>";
                info += "<tr><td>Zoom-Level</td><td>" + browser_.getZoomLevel() + "</td></tr>";
                info += "<tr><td>Using:</td><td>" + MainFrame.version + "</td></tr>";
                info += "</table></body></html>";
                browser_.loadURL(DataUri.create("text/html", info));
            }
        });
        serviceMenu.add(showInfo);

        final JMenuItem showDevTools = new JMenuItem("Show DevTools");
        showDevTools.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                DevToolsDialog devToolsDlg = new DevToolsDialog(owner_, "DEV Tools", browser_);
                devToolsDlg.addComponentListener(new ComponentAdapter() {
                    @Override
                    public void componentHidden(ComponentEvent e) {
                        showDevTools.setEnabled(true);
                    }
                });
                devToolsDlg.setVisible(true);
                devToolsDlg.setSize(740, 695);
                showDevTools.setEnabled(false);
            }
        });
        serviceMenu.add(showDevTools);

        JMenuItem testURLRequest = new JMenuItem("Show URL Request");
        testURLRequest.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                UrlRequestDialog dlg = new UrlRequestDialog(owner_, "URL Request Test");
                dlg.setVisible(true);
                dlg.setSize(740, 695);
            }
        });
        serviceMenu.add(testURLRequest);

        JMenuItem newWindow = new JMenuItem("New window");
        newWindow.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                MainFrame frame;
				try {
					frame = new MainFrame(OS.isLinux(), false, false, null);
					frame.setSize(740, 695);
	                frame.setVisible(true);
				} catch (IOException e1) {
					e1.printStackTrace();
				} catch (URISyntaxException e1) {
					e1.printStackTrace();
				}
            }
        });
        serviceMenu.addSeparator();
        serviceMenu.add(newWindow);

        add(fileMenu);
        add(bookmarkMenu_);
        add(serviceMenu);
    }

    public void addBookmark(String name, String URL) {
        if (bookmarkMenu_ == null) return;

        // Test if the bookmark already exists. If yes, update URL
        Component[] entries = bookmarkMenu_.getMenuComponents();
        for (Component itemEntry : entries) {
            if (!(itemEntry instanceof JMenuItem)) continue;

            JMenuItem item = (JMenuItem) itemEntry;
            if (item.getText().equals(name)) {
                item.setActionCommand(URL);
                return;
            }
        }

        JMenuItem menuItem = new JMenuItem(name);
        menuItem.setActionCommand(URL);
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                browser_.loadURL(e.getActionCommand());
            }
        });
        bookmarkMenu_.add(menuItem);
        validate();
    }

    public void addBookmarkSeparator() {
        bookmarkMenu_.addSeparator();
    }
}
